//===============================================================================
// Microsoft Exception Management Application Block for .NET
// http://msdn.microsoft.com/library/en-us/dnbda/html/emab-rm.asp
//
// BaseApplicationException.cs
// This file contains the definition for the BaseApplicationException.
// For more information see the Implementing the Base Application Exception Class
// section of the Exception Management Application Block Implementation Overview. 
//===============================================================================
// Copyright (C) 2000-2001 Microsoft Corporation
// All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Runtime.Serialization;
using System.Reflection;
using System.Configuration;
using System.Diagnostics;
using System.Threading;
using System.Collections;
using System.Text;
using System.Security;
using System.Security.Principal;
using System.Security.Permissions;
using System.Collections.Specialized;
using System.Resources;

namespace gov.va.med.vbecs.ExceptionManagement
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/15/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// Base Application Exception Class. You can use this as the base exception object from
	/// which to derive your applications exception hierarchy.
	///</summary>

	#endregion

	[Serializable]
	public class BaseApplicationException : ApplicationException
	{
		#region Constructors


		private const string MACHINE_NAME = "machineName";
		private const string CREATED_DATETIME = "createdDateTime";
		private const string APP_DOMAIN = "appDomainName";
		private const string THREAD_IDENTITY = "threadIdentity";
		private const string WINDOWS_IDENTITY = "windowsIdentity";
		private const string ADDITIONAL_INFO = "additionalInformation";

		private const string PERMISSION_DENIED = "RES_EXCEPTIONMANAGEMENT_PERMISSION_DENIED";
		private const string ACCESS_EXCEPTION = "RES_EXCEPTIONMANAGEMENT_INFOACCESS_EXCEPTION";


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///		<Case type="0" testid ="2028"> 
		///			<ExpectedInput>None</ExpectedInput>
		///			<ExpectedOutput>A valid BaseApplicationException object</ExpectedOutput>
		///		</Case>
		///		<Case type="1" testid ="514"> 
		///			<ExpectedInput>NA</ExpectedInput>
		///			<ExpectedOutput>NA</ExpectedOutput>
		///		</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor with no params.
		/// </summary>
		public BaseApplicationException() : base()
		{
			InitializeEnvironmentInformation();
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="2029"> 
		///		<ExpectedInput>A valid string message</ExpectedInput>
		///		<ExpectedOutput>A valid BaseApplicationException object with a message set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2033"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor allowing the Message property to be set.
		/// </summary>
		/// <param name="message">String setting the message of the exception.</param>
		public BaseApplicationException(string message) : base(message) 
		{
			InitializeEnvironmentInformation();
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2031"> 
		///		<ExpectedInput>A valid string message, A valid inner exception</ExpectedInput>
		///		<ExpectedOutput>A valid BaseApplicationException object with a message set and an inner exception set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2040"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor allowing the Message and InnerException property to be set.
		/// </summary>
		/// <param name="message">String setting the message of the exception.</param>
		/// <param name="inner">Sets a reference to the InnerException.</param>
		public BaseApplicationException(string message,Exception inner) : base(message, inner)
		{
			InitializeEnvironmentInformation();
		}

		/// <summary>
		/// Constructor used for deserialization of the exception class.
		/// </summary>
		/// <param name="info">Represents the SerializationInfo of the exception.</param>
		/// <param name="context">Represents the context information of the exception.</param>
		protected BaseApplicationException(SerializationInfo info, StreamingContext context) : base(info, context)
		{
			machineName = info.GetString(MACHINE_NAME);
			createdDateTime = info.GetDateTime(CREATED_DATETIME);
			appDomainName = info.GetString(APP_DOMAIN);
			threadIdentity = info.GetString(THREAD_IDENTITY);
			windowsIdentity = info.GetString(WINDOWS_IDENTITY);
			additionalInformation = (NameValueCollection)info.GetValue(ADDITIONAL_INFO,typeof(NameValueCollection));
		}
		#endregion

		#region Declare Member Variables
		// Member variable declarations
		private string machineName; 
		private string appDomainName;
		private string threadIdentity; 
		private string windowsIdentity; 
		private DateTime createdDateTime = DateTime.Now;

		private static ResourceManager resourceManager = new ResourceManager("VBECS.Common.ExceptionManagerText",Assembly.GetAssembly(typeof(ExceptionManager)));
		
		// Collection provided to store any extra information associated with the exception.
		private NameValueCollection additionalInformation = new NameValueCollection();
		
		#endregion
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1174"> 
		///		<ExpectedInput>A valid SerializationInfo, A valid StreamingContext</ExpectedInput>
		///		<ExpectedOutput>None - pass</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2041"> 
		///		<ExpectedInput>An invalid SerializationInfo, An invalid StreamingContext</ExpectedInput>
		///		<ExpectedOutput>None - fail</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method overrides the GetObjectData method in Exception.  It sets the SerializationInfo
		/// with information or custom values about the exception.
		/// </summary>
		/// <param name="info">Represents the SerializationInfo of the exception.</param>
		/// <param name="context">Represents the context information of the exception.</param>
		[SecurityPermission(SecurityAction.Demand, SerializationFormatter = true)]
		public override void GetObjectData( SerializationInfo info, StreamingContext context ) 
		{
			info.AddValue(MACHINE_NAME, machineName, typeof(string));
			info.AddValue(CREATED_DATETIME, createdDateTime);
			info.AddValue(APP_DOMAIN, appDomainName, typeof(string));
			info.AddValue(THREAD_IDENTITY, threadIdentity, typeof(string));
			info.AddValue(WINDOWS_IDENTITY, windowsIdentity, typeof(string));
			info.AddValue(ADDITIONAL_INFO, additionalInformation, typeof(NameValueCollection));
			base.GetObjectData(info,context);
		}

		#region Public Properties
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2048"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="519"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Machine name where the exception occurred.
		/// </summary>
		public string MachineName
		{
			get
			{
				return machineName;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2047"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid DateTime</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="520"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Date and Time the exception was created.
		/// </summary>
		public DateTime CreatedDateTime
		{
			get
			{
				return createdDateTime;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2046"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="522"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AppDomain name where the exception occurred.
		/// </summary>
		public string AppDomainName
		{
			get
			{
				return appDomainName;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2045"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="523"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Identity of the executing thread on which the exception was created.
		/// </summary>
		public string ThreadIdentityName
		{
			get
			{
				return threadIdentity;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2044"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="524"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Windows identity under which the code was running.
		/// </summary>
		public string WindowsIdentityName
		{
			get
			{
				return windowsIdentity;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2043"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid NameValueCollection</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="525"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Collection allowing additional information to be added to the exception.
		/// </summary>
		public NameValueCollection AdditionalInformation
		{
			get
			{
				return additionalInformation;
			}
		}
		#endregion

		/// <summary>
		/// Initialization function that gathers environment information safely.
		/// </summary>
		private void InitializeEnvironmentInformation()
		{									
			try
			{				
				machineName = Environment.MachineName;
			}
			catch(SecurityException)
			{
				machineName = resourceManager.GetString(PERMISSION_DENIED);
				
			}
			catch
			{
				machineName = resourceManager.GetString(ACCESS_EXCEPTION);
			}
					
			try
			{
				threadIdentity = Thread.CurrentPrincipal.Identity.Name;
			}
			catch(SecurityException)
			{
				threadIdentity = resourceManager.GetString(PERMISSION_DENIED);
			}
			catch
			{
				threadIdentity = resourceManager.GetString(ACCESS_EXCEPTION);
			}			
			
			try
			{
				windowsIdentity = WindowsIdentity.GetCurrent().Name;
			}
			catch(SecurityException)
			{
				windowsIdentity = resourceManager.GetString(PERMISSION_DENIED);
			}
			catch
			{
				windowsIdentity = resourceManager.GetString(ACCESS_EXCEPTION);
			}
			
			try
			{					
				appDomainName = AppDomain.CurrentDomain.FriendlyName;
			}
			catch(SecurityException)
			{
				appDomainName = resourceManager.GetString(PERMISSION_DENIED);
			}
			catch
			{
				appDomainName = resourceManager.GetString(ACCESS_EXCEPTION);
			}			
		}
	}
}

